/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines tools/types for T=1 protocol handling
 *
 * Author :         Andreas Schwier (ASC)
 *
 * Last modified:   21.Jan 2000
 *
 *****************************************************************************/

#ifdef DEBUG
#include <stdio.h>
#endif

#include <stdlib.h>
#include <string.h>

#include "sercom.h"
#include "eco5000.h"
#include "ecotools.h"
#include "ecoT1.h"
#include "ctapi.h"


/*
 * Initialize all T=1 protocol parameter to its default values
 *
 */

void ecoT1InitProtocol(struct eco5000_t *ctx)

{
    ctx->Data.t1->WorkBWT = ctx->Data.t1->BlockWaitTime;
    ctx->Data.t1->IFSC = ctx->T1_IFSC;
    ctx->Data.t1->SSequenz = 0;
    ctx->Data.t1->RSequenz = 0;
}



/*
 * Initialize T=1 protocol driver module
 *
 */

int ecoT1Init (struct eco5000_t *ctx)

{
    int rc;

    ctx->Data.t1 = malloc(sizeof(struct ecoT1_t));

    ctx->CTModFunc = (CTModFunc_t) ecoT1Process;

    ctx->Data.t1->BlockWaitTime = 200 + (1 << ctx->T1_BWI) * 100 + 11000 / ctx->Baud;
    ctx->Data.t1->CharWaitTime  = 100 + ((11 + (1 << ctx->T1_CWI)) * 1000 / ctx->Baud);

    ecoT1InitProtocol(ctx);

    if ((rc = ecoCommand(ctx, SET_T1, 0, NULL, 0, NULL)) < 0)
        return rc;

    if ((rc = ecoCommand(ctx, RX_ON, 0, NULL, 0, NULL)) < 0)
        return rc;

    /* ecoT1Resynch(ctx, 0, 0); */

    return 0;
}



/*
 * Terminate driver module and release memory
 *
 */

int ecoT1Term (struct eco5000_t *ctx)

{
    free(ctx->Data.t1);
    ctx->Data.t1 = NULL;
    ctx->CTModFunc = NULL;
    return 0;
}



#ifdef DEBUG
/*
 * Print the block information of a T=1 transmission block
 *
 */

void ecoT1BlockInfo(byte Nad, byte Pcb, int InBuffLength, byte *InBuff)

{
    byte *ptr;
    int cnt;

    printf("SAD:%02d-DAD:%02d  ", SAD(Nad), DAD(Nad));

    if (ISIBLOCK(Pcb)) {
        printf("I(%d,%d):", NS(Pcb), MORE(Pcb));
        ptr = InBuff;
        for (cnt = 0; cnt < InBuffLength; cnt++) {
            printf("%02X ",(int)*ptr);
            ptr++;
        }
        printf("\n");
    }

    if (ISRBLOCK(Pcb))
        printf("R(%d)[%d]\n", NR(Pcb), RERR(Pcb));

    if (ISSBLOCK(Pcb))
        switch (SBLOCKFUNC(Pcb)) {
            case RESYNCHREQ :
                printf("S(RESYNCH Request)\n"); break;
            case RESYNCHRES :
                printf("S(RESYNCH Response)\n"); break;
            case IFSREQ :
                printf("S(IFS Request:%d)\n",(int)InBuff[0]); break;
            case IFSRES :
                printf("S(IFS Response:%d)\n",(int)InBuff[0]); break;
            case ABORTREQ       :
                printf("S(ABORT Request)\n"); break;
            case ABORTRES       :
                printf("S(ABORT Response)\n"); break;
            case WTXREQ :
                printf("S(WTX Request:%d)\n",(int)InBuff[0]); break;
            case WTXRES :
                printf("S(WTX Response:%d)\n",(int)InBuff[0]); break;
            default :
                printf("Unknown S(...) Block\n");
        }
}
#endif



/*
 * Receive a block in T=1 protocol
 *
 */

int ecoT1ReceiveBlock(struct eco5000_t *ctx)

{
    int ret,Len,BuffLen;
    byte *ptr,lrc,rch;

    rs232Mode(ctx->fh, -1, 0, -1, -1, ctx->Data.t1->WorkBWT);

    ctx->Data.t1->InBuffLength = -1;

    if ((ret = iccRead(ctx->fh, ctx->Indirect, &rch, 1)) != 1) { /* Receive NAD byte              */
#ifdef DEBUG
        printf("Error waiting for first character rc=%d.\n", ret);
#endif
        return(ret < 0 ? ret : ERR_TIMEOUT);
    }

    rs232Mode(ctx->fh, -1, 0, -1, -1, ctx->Data.t1->CharWaitTime);

    lrc  = ctx->Data.t1->Nad = rch;

    if ((ret = iccRead(ctx->fh, ctx->Indirect, &rch, 1)) != 1)  /* Receive PCB byte              */
        return(ret < 0 ? ret : ERR_TIMEOUT);

    lrc ^= ctx->Data.t1->Pcb = rch;

    if ((ret = iccRead(ctx->fh, ctx->Indirect, &rch, 1)) != 1)  /* Receive LEN byte              */
        return(ret < 0 ? ret : ERR_TIMEOUT);

    lrc ^= rch;                         /* Calculate LRC                     */
    Len = ctx->Data.t1->InBuffLength = rch;

    ptr = ctx->Data.t1->InBuff;
    BuffLen = BUFFMAX;

    while (Len && BuffLen--) {
        if ((ret = iccRead(ctx->fh, ctx->Indirect, &rch, 1)) != 1)
            return(ret < 0 ? ret : ERR_TIMEOUT);

        lrc ^= *ptr++ = rch;
        Len--;
    }

    if (Len)
        return(-1);

    if ((ret = iccRead(ctx->fh, ctx->Indirect, &rch, 1)) != 1)
        return(ret < 0 ? ret : ERR_TIMEOUT);

    ret = lrc == rch ? 0 : ERR_EDC;

#ifdef DEBUG
    printf("Received: lrc=%02X rch=%02X ", lrc, rch);
    ecoT1BlockInfo(ctx->Data.t1->Nad, ctx->Data.t1->Pcb, ctx->Data.t1->InBuffLength, ctx->Data.t1->InBuff);
#endif

    return(ret);
}



/*
 * Send a block in T=1 protocol
 *
 */

int ecoT1SendBlock(struct eco5000_t *ctx,
                   byte Nad,
                   byte Pcb,
                   byte *Buffer,
                   int BuffLen)

{
    int ret,len;
    byte sndbuf[BUFFMAX];
    byte *ptr, lrc;

    if (BuffLen > 254)
        return(-1);

    ptr = sndbuf;
    *ptr++ = Nad;
    *ptr++ = Pcb;
    *ptr++ = (byte)BuffLen;
    memcpy(ptr, Buffer, BuffLen);

    lrc = 0;
    for (len = BuffLen + 3, ptr = sndbuf; len; len--, ptr++)
        lrc ^= *ptr;

    *ptr = lrc;

    iccWrite(ctx->fh, ctx->Indirect, sndbuf, BuffLen + 4);

    len = BuffLen + 4;
    while (len > 0) {
        ret = iccRead(ctx->fh, ctx->Indirect, sndbuf, len);
        if (ret < 0)
            return ret;
        len -= ret;
    }

#ifdef DEBUG
    printf("Sending : ");
    ecoT1BlockInfo(Nad, Pcb, BuffLen, Buffer);
#endif

    return(0);
}



/*
 * Synchronize sequence counter in both sender and receiver
 * after a transmission error has occured
 *
 */

int ecoT1Resynch(struct eco5000_t *ctx, int SrcNode, int DestNode)

{
    int ret,retry;

    retry = RETRY;

    while (retry--) {
        ret = ecoT1SendBlock(ctx,
                             CODENAD(SrcNode, DestNode),
                             CODESBLOCK(RESYNCHREQ),
                             NULL,0);

        if (ret < 0)
            return(-1);

        ret = ecoT1ReceiveBlock(ctx);

        if (!ret &&
            ISSBLOCK(ctx->Data.t1->Pcb) &&
            (SBLOCKFUNC(ctx->Data.t1->Pcb) == RESYNCHRES)) {
            ecoT1InitProtocol(ctx);
            return(0);
        }
    }
    return(-1);
}



/* 
 * Abort a sequence of chained transmission blocks
 *
 */

int ecoT1AbortChain(struct eco5000_t *ctx, int SrcNode, int DestNode)


{
    int ret,retry;

    retry = RETRY;

    while (retry--) {
        ret = ecoT1SendBlock(ctx,
                             CODENAD(SrcNode, DestNode),
                             CODESBLOCK(ABORTREQ),
                             NULL, 0);

        if (ret < 0)
            return(-1);

        ret = ecoT1ReceiveBlock(ctx);

        if (!ret && ISSBLOCK(ctx->Data.t1->Pcb) && (SBLOCKFUNC(ctx->Data.t1->Pcb) == ABORTRES))
            return(0);
    }

    return(-1);
}



/*
 * Receive a transmission block and handle all S-block requests
 *
 */

int ecoT1GetBlock(struct eco5000_t *ctx, int SrcNode, int DestNode)

{
    int retry,ret;

    retry = RETRY;
    ctx->Data.t1->WorkBWT = ctx->Data.t1->BlockWaitTime;

    while (TRUE) {
        ret = ecoT1ReceiveBlock(ctx);

        if (ret < 0) {
            if (!retry)
                return(ret);

            /* The receiver did not understand our transmission block.             */
            /* Unless the retry counter expires, we send it again                  */

            retry--;
            rs232Drain(ctx->fh);
            rs232Flush(ctx->fh);
            ret = ecoT1SendBlock(ctx,
                                 CODENAD(SrcNode, DestNode),
                                 CODERBLOCK(ctx->Data.t1->RSequenz, ret == ERR_EDC ? 1 : 2),
                                 NULL,
                                 0);

            if (ret < 0)
                return(ret);

            ctx->Data.t1->WorkBWT = ctx->Data.t1->BlockWaitTime;
            continue;
        }

        /* Lets see, if we received a S-block from the other side                */

        if (ISSBLOCK(ctx->Data.t1->Pcb)) {
            switch(SBLOCKFUNC(ctx->Data.t1->Pcb)) {
                case RESYNCHRES :               /* Request to synchronize again      */
                    ecoT1InitProtocol(ctx);
                    return(1);
                    break;

                case IFSREQ :                   /* Request to change the buffer size */
                    ecoT1SendBlock(ctx,
                                   CODENAD(SrcNode, DestNode),
                                   CODESBLOCK(IFSRES),
                                   ctx->Data.t1->InBuff,
                                   1);
                    ctx->Data.t1->IFSC = (int)ctx->Data.t1->InBuff[0];

#ifdef DEBUG
                    printf("New IFSC: %d bytes.\n", ctx->Data.t1->IFSC);
#endif
                    break;

                case ABORTREQ :                 /* Request to abort the transmission */
                    ecoT1SendBlock(ctx,
                                   CODENAD(SrcNode, DestNode),
                                   CODESBLOCK(ABORTRES),
                                   NULL,
                                   0);

                    ecoT1ReceiveBlock(ctx);
                    return(-1);
                    break;

                case WTXREQ :                   /* Request to extend timeout         */
                    ecoT1SendBlock(ctx,
                                   CODENAD(SrcNode, DestNode),
                                   CODESBLOCK(WTXRES),
                                   ctx->Data.t1->InBuff,
                                   1);
                    ctx->Data.t1->WorkBWT = ctx->Data.t1->BlockWaitTime *
                        (int)ctx->Data.t1->InBuff[0];

#ifdef DEBUG
                    printf("New BWT value %ld ms.\n",ctx->Data.t1->WorkBWT);
#endif
                    break;

                default :
                    return(-1);
            }
        }

        if (ISRBLOCK(ctx->Data.t1->Pcb) || ISIBLOCK(ctx->Data.t1->Pcb))
            break;
    }

    return(0);
}



/*
 * Send a block of data using T=1 protocol
 * Handle large block with the chaining mechanism
 *
 */

int ecoT1SendData(struct eco5000_t *ctx,
                  int HostMode,
                  int SrcNode,
                  int DestNode,
                  byte *Buffer,
                  int BuffLen)

{
    int ret,more,retry,Length,response;

    while (BuffLen) {
        Length = MIN(BuffLen, ctx->Data.t1->IFSC);
        BuffLen -= Length;
        more = BuffLen ? 1 : 0;
    
        retry = RETRY;
        response = 0;
    
        while (response <= 0) {
            ret = ecoT1SendBlock(ctx,
                                 CODENAD(SrcNode, DestNode),
                                 CODEIBLOCK(ctx->Data.t1->SSequenz,more),
                                 Buffer,
                                 Length);

            if (ret < 0)
                return(ret);


            if (!more & HostMode) {
                ctx->Data.t1->SSequenz = 1 - ctx->Data.t1->SSequenz;
                return(0);
            }

            ret = ecoT1GetBlock(ctx, SrcNode, DestNode);

            if (ret < 0)                      /* Something went wrong              */
                return(ret);

            if (ret > 0)                      /* Send block again                  */
                continue;

            /* A block can be acknowledged with an I or R-block                    */ 

            if (ISRBLOCK(ctx->Data.t1->Pcb)) {
                if (NR(ctx->Data.t1->Pcb) == ctx->Data.t1->SSequenz) {
                    if (retry--)
                        continue;                   /* Receiver is requesting same block */

                    if (ecoT1Resynch(ctx,SrcNode,DestNode))
                        return(-1);
                    continue;
                } else if (more) {

                    /* In chaining mode, the R-block is used to acknowledge the I-block*/

                    response = 1;
                    ctx->Data.t1->SSequenz = 1 - ctx->Data.t1->SSequenz;
                } else {
#ifdef DEBUG
                    printf("Unexpected R-Block.\n");
#endif
                    if (ecoT1Resynch(ctx,SrcNode,DestNode))
                        return(-1);
                }
            }

            if (ISIBLOCK(ctx->Data.t1->Pcb)) {    /* Usual response to I-block     */
                if ((NS(ctx->Data.t1->Pcb) != ctx->Data.t1->RSequenz) || more) {
#ifdef DEBUG
                    printf("Asynchronous I-Block received as response.\n");
#endif
                    if (ecoT1Resynch(ctx,SrcNode,DestNode))
                        return(-1);
                } else {
                    ctx->Data.t1->SSequenz = 1 - ctx->Data.t1->SSequenz;
                    response = 1;
                }
            }
        }

        Buffer += Length;
    }

    return(0);
}



/*
 * Decode a received block into the data buffer passed to the application
 *
 */

int ecoT1ReceiveData(struct eco5000_t *ctx,
                     int SrcNode,
                     int DestNode,
                     byte *Buffer,
                     int BuffLen)
{
    int more,Length,ret;

    Length = 0;
    do  {

        if ((ctx->Data.t1->InBuffLength > BuffLen) || (ctx->Data.t1->InBuffLength == -1)) {
            ecoT1AbortChain(ctx,SrcNode,DestNode);
            return(ERR_MEMORY);                       /* Out of space in buffer            */
        }

        memcpy(Buffer, ctx->Data.t1->InBuff, ctx->Data.t1->InBuffLength);
        Buffer += ctx->Data.t1->InBuffLength;
        BuffLen -= ctx->Data.t1->InBuffLength;
        Length += ctx->Data.t1->InBuffLength;

        ctx->Data.t1->RSequenz = 1 - ctx->Data.t1->RSequenz;

        more = MORE(ctx->Data.t1->Pcb);     /* More block following ?            */

        if (more) {
            while (TRUE) {
                ret = ecoT1SendBlock(ctx,
                                     CODENAD(SrcNode, DestNode),
                                     CODERBLOCK(ctx->Data.t1->RSequenz, 0),
                                     NULL, 0);

                ret = ecoT1GetBlock(ctx, SrcNode, DestNode);

                if (ret < 0)
                    return(ret);

                if (ISRBLOCK(ctx->Data.t1->Pcb)) {
                    if (NR(ctx->Data.t1->Pcb) != ctx->Data.t1->SSequenz) {

#ifdef DEBUG
                        printf("Invalid sequenz received in R-Block.\n");
#endif

                        if (ecoT1Resynch(ctx,SrcNode,DestNode))
                            return(-1);
                    }
                    continue;
                }

                break;
            }
        }
    } while (more);

    return(Length);
}



/*
 * Transport a data block using T=1 transmission protocol
 *
 */

int ecoT1Transport(struct eco5000_t *ctx,
                   int SrcNode,
                   int DestNode,
                   byte *OBuffer,
                   int OBuffLen,
                   byte *IBuffer,
                   int IBuffLen)

{
    unsigned char case0[4];
  
    int ret;

    /* To allow correct mapping for Case 0 APDU's to that are passed
     * with a size less than 4 byte, a new APDU block is created
     */

    rs232Flush(ctx->fh);
    if (OBuffLen < 4) {
        memset(case0, 0, sizeof(case0));
        memcpy(case0, OBuffer, OBuffLen);
        ret = ecoT1SendData(ctx, FALSE, SrcNode, DestNode, case0, 4);
    } else
        ret = ecoT1SendData(ctx, FALSE, SrcNode, DestNode, OBuffer, OBuffLen);

    if (ret < 0)
        return(ret);

    ret = ecoT1ReceiveData(ctx, SrcNode, DestNode, IBuffer, IBuffLen);
    return(ret);
}



/*
 * Process a APDU using T=1 protocol
 *
 */

int ecoT1Process (struct eco5000_t *ctx,
                  unsigned int  lc, 
                  unsigned char *cmd, 
                  unsigned int  *lr,
                  unsigned char *rsp)

{
    int rc;

    ecoCommand(ctx, RX_ON, 0, NULL, 0, NULL);

    rc = ecoT1Transport(ctx, 0, 0, cmd, lc, rsp, *lr);

    if (rc < 0)
        return rc;

    *lr = rc;
    return 0;
}
