/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Provide low level serial communication functions
 *
 * Author :         Andreas Schwier (ASC)
 *
 * Last modified:   21.Jan 2000
 *
 *****************************************************************************/

#include <stdlib.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include <unistd.h>
#include <termios.h>


#include "sercom.h"


#ifdef DEBUG
static void dump(char *tag, int want, unsigned char *p, int len)

{
    printf("%s(%d) got %d: ", tag, want, len);
    while (len > 0) {
	printf("%02X ", *p);
	len--;
	p++;
    }
    printf("\n");
}
#endif



static speed_t rs232Baud2Speed(long baud)

{
    speed_t rc;

    switch(baud) {
	case 300:       rc = B300; break;
	case 2400:      rc = B2400; break;
	case 4800:      rc = B4800; break;
	case 9600:      rc = B9600; break;
	case 19200:     rc = B19200; break;
	case 38400:     rc = B38400; break;
#ifdef B57600
	case 57600:     rc = B57600; break;
#endif
#ifdef B115200
	case 115200:    rc = B115200; break;
#endif
#ifdef B230400
	case 230400:    rc = B230400; break;
#endif
#ifdef B460800
	case 460800:    rc = B460800; break;
#endif
	default:        return 0;
    }
    return rc;
}



/*
 * Change a communication line parameter
 *
 * If a parameter is set to -1, it remains unchanged
 *
 * Returns      ERR_RS232OPEN           if port is not open
 *              ERR_RS232PARAM          if invalid parameter provided
 */

int rs232Mode(HANDLE fh, long baud, char parity, int bits, int stopbits, long timeout)

{
    int options;
    struct termios tio;

    options = TCSANOW;

    if (tcgetattr(fh, &tio) < 0)          /* Get current configuration         */
	return ERR_RS232OPEN;

    if (baud != -1) {                     /* Change baudrate if specified      */
	options = TCSAFLUSH;
	if (cfsetospeed(&tio, rs232Baud2Speed(baud)) < 0)
	    return ERR_RS232PARAM;

	if (cfsetispeed(&tio, 0) < 0)       /* Both speeds are equal             */
	    return ERR_RS232PARAM;
    }

    if (parity != -1) {                   /* Change parity if specified        */
	options = TCSAFLUSH;
	parity = toupper(parity);
	tio.c_cflag &= ~(PARENB|PARODD|INPCK);

	if (parity != 'N') {
	    tio.c_cflag |= PARENB|INPCK;      /* Enable parity checking            */
	    if (parity == 'O')
		tio.c_cflag |= PARODD;          /* Set odd parity                    */
	    else if (parity != 'E')
		return ERR_RS232PARAM;
	}
    }

    if (bits != -1) {                     /* Change bits if specified          */
	options = TCSAFLUSH;
	tio.c_cflag &= ~CSIZE;

	switch(bits) {
	    case 7:         tio.c_cflag |= CS7; break;
	    case 8:         tio.c_cflag |= CS8; break;
	    default:
		return ERR_RS232PARAM;
	}
    }

    if (stopbits != -1) {
	options = TCSAFLUSH;
	tio.c_cflag &= ~CSTOPB;
	if (stopbits != 1)
	    tio.c_cflag |= CSTOPB;
    }

    if (timeout != -1) {                  /* Timeout is in ms                  */
	memset(tio.c_cc, 0, sizeof(tio.c_cc));
	if (timeout < 25500)
	    tio.c_cc[VTIME] = timeout / 100;
	else
	    tio.c_cc[VTIME] = 255;
    }

    tio.c_cflag |= CREAD|HUPCL|CLOCAL;
    tio.c_cflag &= ~CRTSCTS; 

    tio.c_iflag = 0;
    tio.c_oflag = 0;  
    tio.c_lflag = 0; 
	
    tio.c_cc[VMIN]  = 0;

    if (tcsetattr(fh, options, &tio) < 0) {
	return ERR_RS232PARAM;
    }

    return 0;
}



/*
 * Open a serial port with the parameter specified
 *
 * Returns      ERR_RS232OPEN           if port can not be opened
 *              ERR_RS232PARAM          if invalid parameter provided
 */

int rs232Open(HANDLE *fh, char *port, long baud, char parity, int bits, int stopbits, long timeout)

{
    int fd,rc;
  
    fd = open(port, O_RDWR | O_NOCTTY);

    if (fd < 0)
	return ERR_RS232OPEN;

    rc = rs232Mode(fd, baud, parity, bits, stopbits, timeout);
    if (rc < 0) {
	close(fd);
	return rc;
    }

    rs232LineControl(fd, 0, 0);           /* Set DTR and RTS off               */
  
    rs232Flush(fd);
    rs232Drain(fd);

    *fh = fd;

    return 0;
}



/*
 * Control the status of the DTR and RTS line
 *
 */

int rs232LineControl(HANDLE fh, int dtr, int rts)

{
    unsigned int mctl;

    if (ioctl(fh, TIOCMGET, &mctl) < 0)
	return ERR_RS232OPEN;

    if (dtr) {
	mctl |=TIOCM_DTR;
#ifdef DEBUG
	printf("\n[rs232lineControl]  Setting DTR ON\n");
#endif
    }
    else {
	mctl &=~TIOCM_DTR;
#ifdef DEBUG
	printf("\n[rs232lineControl]  Setting DTR OFF\n");
#endif
    }
    
    if (rts) {
	mctl |=TIOCM_RTS;
#ifdef DEBUG
	printf("\n[rs232lineControl]  Setting RTS ON\n");
#endif
    }    
    else {
	mctl &=~TIOCM_RTS;
#ifdef DEBUG
	printf("\n[rs232lineControl]  Setting RTS OFF\n");
#endif
    }
    
    if (ioctl(fh, TIOCMSET, &mctl) < 0)
	return ERR_RS232OPEN;

    return 0;
}



/*
 * Test the status of the DSR and CTS signal
 *
 */

int rs232LineTest(HANDLE fh, int *dsr, int *cts)

{
    unsigned int mctl;

    if (ioctl(fh, TIOCMGET, &mctl) < 0)
	return ERR_RS232OPEN;

    *dsr = mctl & TIOCM_DSR ? 1 : 0;
    *cts = mctl & TIOCM_CTS ? 1 : 0;

    return 0;
}



/*
 * Read a block of character from the communication line
 *
 * Return the number of character read
 *
 */

int rs232Read(HANDLE fh, unsigned char *buff, int len)

{
    int rc,rm;
    unsigned char *p;

    rm = len;
    p = buff;

    do {
	rc = read(fh, p, rm);

	if (rc < 0)
	    return rc;

	rm -= rc;
	p += rc;
    } while (rc && rm);

#ifdef DEBUG
    dump("READ", len, buff, len - rm);
#endif

    return len - rm;
}



/* 
 * Write a block of character to the communication line
 *
 * Return the number of character written
 *
 */

int rs232Write(HANDLE fh, unsigned char *buff, int len)

{
    int rc;

    rc = write(fh, buff, len);

#ifdef DEBUG
    dump("WRITE", len, buff, rc);
#endif

    return rc;
}



/*
 * Drain all characters remaining in the outgoing data queue
 *
 */

int rs232Drain(HANDLE fh)

{
    return tcdrain(fh);
}



/*
 * Flush all characters remaining in the incoming data queue
 *
 */

int rs232Flush(HANDLE fh)

{
    return tcflush(fh, TCIOFLUSH);
}



/*
 * Close a communication channel
 *
 */

int rs232Close(HANDLE fh)

{
    return close(fh);
}




